/* ValueIterator.cpp
 *
 * jRegistryKey - A JNI wrapper of the Windows Registry functions.
 * Copyright (c) 2001, BEQ Technologies Inc.
 * #205, 3132 Parsons Road
 * Edmonton, Alberta
 * T6N 1L6 Canada
 * (780) 430-0056
 * (780) 437-6121 (fax)
 * http://www.beq.ca
 *
 * Original Author: Joe Robinson <joe@beq.ca>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Joe Robinson <joe@beq.ca>
 * Version: $Revision: 1.1.1.1 $
 *
 * $Log: ValueIterator.cpp,v $
 * Revision 1.1.1.1  2001/10/25 14:58:36  joe.robinson
 * Added project to CVS
 *
 */
#include <windows.h>

#include "jniClassDescriptor.h"
#include "jniFieldDescriptor.h"
#include "jRegistryKey.h"
#include "ca_beq_util_win32_registry_ValueIterator.h"

extern int getRootKey(JNIEnv *env, jobject obj);

/**
 * startValueIteration()
 *    Helper method to set up RegistryValue enumeration.
 */
static int startValueIteration(JNIEnv* env, jobject obj) {
   // get KeyIterator class description
   jclass CValueIterator = env->GetObjectClass(obj);
   if(CValueIterator == NULL) { return -1; }

   // get field ID of RegistryKey (KeyIterator.key) object from KeyIterator object
   jfieldID fid_key = env->GetFieldID(CValueIterator, "key", registryDescriptor::jfd_RegistryKey);
   if(fid_key == NULL) { return -1; }

   // get reference to RegistryKey object from KeyIterator object
   jobject obj_RegistryKey = env->GetObjectField(obj, fid_key);
   if(obj_RegistryKey == NULL) { return -1; }

   // get root key from RegistryKey object
   int nRootKey = getRootKey(env, obj_RegistryKey);

   // get RegistryKey class description
   jclass CRegistryKey = env->GetObjectClass(obj_RegistryKey);
   if(CRegistryKey == NULL) { return -1; }

   // get field ID path member of RegistryKey
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return -1; }

   // get path member from RegistryKey object
   jstring path = (jstring)env->GetObjectField(obj_RegistryKey, fid_path);
   const char *szPath = env->GetStringUTFChars(path, NULL);

   // open the registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Open key failed");
      env->ReleaseStringUTFChars(path, szPath);
      return -1;
   } // if
   env->ReleaseStringUTFChars(path, szPath);

   // query count and max length of keys
   int count = 0;
   int maxsize = 0;
   if(RegQueryInfoKey(hkey, NULL, NULL, NULL, NULL, NULL, NULL, (LPDWORD)&count, (LPDWORD)&maxsize, NULL, NULL, NULL) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Query info key failed");
      return -1;
   } // if

   // save hkey reference
   jfieldID fid_hkey = env->GetFieldID(CValueIterator, "hkey", fieldDescriptor::jfd_int);
   if(fid_hkey == NULL) { return 0; }
   env->SetIntField(obj, fid_hkey, (DWORD)hkey);

   // save maximum size of key entry
   jfieldID fid_maxsize = env->GetFieldID(CValueIterator, "maxsize", fieldDescriptor::jfd_int);
   if(fid_maxsize == NULL) { return 0; }
   env->SetIntField(obj, fid_maxsize, maxsize + 1);

   // save current index
   jfieldID fid_index = env->GetFieldID(CValueIterator, "index", fieldDescriptor::jfd_int);
   if(fid_index == NULL) { return 0; }
   env->SetIntField(obj, fid_index, 0);

   // save number of items
   jfieldID fid_count = env->GetFieldID(CValueIterator, "count", fieldDescriptor::jfd_int);
   if(fid_count == NULL) { return 0; }
   env->SetIntField(obj, fid_count, count);

   return count;
} // startValueIteration()

/**
 * ValueIterator::hasNext()
 *    Tests if the registry key has more values.
 */
JNIEXPORT jboolean JNICALL Java_ca_beq_util_win32_registry_ValueIterator_hasNext(JNIEnv *env, jobject obj) {
   // get the class
   jclass CValueIterator = env->GetObjectClass(obj);
   if(CValueIterator == NULL) { return false; }

   jfieldID fid_index = env->GetFieldID(CValueIterator, "index", fieldDescriptor::jfd_int);
   if(fid_index == NULL) { return false; }
   int index = env->GetIntField(obj, fid_index);

   int count;
   if(index == -1)  {
      // first time
      count = startValueIteration(env, obj);
      index = 0;
   } // if
   else {
      jfieldID fid_count = env->GetFieldID(CValueIterator, "count", fieldDescriptor::jfd_int);
      if(fid_count == NULL) { return false; }
      count = env->GetIntField(obj, fid_count);
   } // else

   return index < count;
} // ValueIterator::hasNext()

/**
 * ValueIterator::getNext()
 *    Retrieves the NAME of the next value from the registry key.
 */
JNIEXPORT jstring JNICALL Java_ca_beq_util_win32_registry_ValueIterator_getNext(JNIEnv *env, jobject obj) {
   // get the class
   jclass CValueIterator = env->GetObjectClass(obj);
   if(CValueIterator == NULL) { return NULL; }

   // get the field IDs
   jfieldID fid_index = env->GetFieldID(CValueIterator, "index", fieldDescriptor::jfd_int);
   if(fid_index == NULL) { return NULL; }
   int index = env->GetIntField(obj, fid_index);

   int count;
   if(index == -1)  {
      // first time
      count = startValueIteration(env, obj);
      index = 0;
   } // if
   else {
      jfieldID fid_count = env->GetFieldID(CValueIterator, "count", fieldDescriptor::jfd_int);
      if(fid_count == NULL) { return NULL; }
      count = env->GetIntField(obj, fid_count);
   } // else

   if(index >= count) {
      // already at end
      env->ThrowNew(env->FindClass(classDescriptor::jcd_utilNoSuchElementException), "past end of enumeration");
      return NULL;
   } // if

   jfieldID fid_maxsize = env->GetFieldID(CValueIterator, "maxsize", fieldDescriptor::jfd_int);
   if(fid_maxsize == NULL) { return NULL; }
   int maxsize = env->GetIntField(obj, fid_maxsize);

   jfieldID fid_hkey = env->GetFieldID(CValueIterator, "hkey", fieldDescriptor::jfd_int);
   if(fid_hkey == NULL) { return NULL; }
   HKEY hkey = (HKEY)env->GetIntField(obj, fid_hkey);
   char *cret = (char*)malloc(maxsize);

   // find the next key
   if(RegEnumValue(hkey, index, cret, (LPDWORD)&maxsize, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Enum value failed");
      free(cret);
      RegCloseKey(hkey);
      env->SetIntField(obj, fid_index, count);
      return NULL;
   } // if

   jstring ret = env->NewStringUTF(cret);
   free(cret);

   // increment index
   index++;
   env->SetIntField(obj, fid_index, index);

   if(index == count) {
      // at end
      RegCloseKey(hkey);
   } // if

   return ret;
} // ValueIterator::getNext()
