/* RegistryKey.cpp
 *
 * jRegistryKey - A JNI wrapper of the Windows Registry functions.
 * Copyright (c) 2001, BEQ Technologies Inc.
 * #205, 3132 Parsons Road
 * Edmonton, Alberta
 * T6N 1L6 Canada
 * (780) 430-0056
 * (780) 437-6121 (fax)
 * http://www.beq.ca
 *
 * Original Author: Joe Robinson <joe@beq.ca>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Joe Robinson <joe@beq.ca>
 * Version: $Revision: 1.1.1.1 $
 *
 * $Log: RegistryKey.cpp,v $
 * Revision 1.1.1.1  2001/10/25 14:58:36  joe.robinson
 * Added project to CVS
 *
 */
#include <string.h>
#include <stdlib.h>
#include <windows.h>
#include <shlwapi.h>

#include "jniClassDescriptor.h"
#include "jniFieldDescriptor.h"
#include "jRegistryKey.h"
#include "ca_beq_util_win32_registry_RegistryKey.h"

// Runtime version identifier
const char RCS[] = "$Id: RegistryKey.cpp,v 1.1.1.1 2001/10/25 14:58:36 joe.robinson Exp $";

/**
 * ::ThrowException()
 *   Helper function that allows us to throw specific Java exception classes by name.
 *
 * @param environment   Java environment interface pointer
 * @param exception     String ID of a Java java.lang.Exception or java.lang.RuntimeException sub-class
 * @param message       Exception detail message
 */
void ThrowException(JNIEnv *environment, const char *exception, const char *message) {
   // lookup java Exception class name
   jclass exceptionClass = environment->FindClass(exception);

   // if exceptionClass is NULL, another exception has been thrown; otherwise, throw our exception
   if(exceptionClass != NULL) {
      environment->ThrowNew(exceptionClass, message);
   } // if

   environment->DeleteLocalRef(exceptionClass);
} // ThrowException()

/**
 * ::JNI_OnLoad()
 *    Invoked by the Java Virtual Machine when the native library is loaded by
 *    the System.loadLibrary() call.
 * @return
 */
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *jvm, void *reserved) {
   // check that we're running JNI 1.2 and if so, get JNIEnv reference
   JNIEnv *env;
   if(jvm->GetEnv((void**)&env, JNI_VERSION_1_2)) {
      return JNI_ERR;
   } // if

   jclass cls = env->FindClass(registryDescriptor::jcd_RegistryKey);
   if(cls == NULL) {
      return JNI_ERR;
   } // if

   return JNI_VERSION_1_2;
} // JNI_OnLoad()

/**
 * JNI_OnUnload()
 *    Invoked by the Java Virtual Machine when the native library is unloaded.
 */
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *jvm, void *reserved) {
   return;
} // JNI_OnUnload()

/**
 * ::getRootKey()
 *   Helper function that retreives the root key from a RegistryKey object.
 * @return
 */
int getRootKey(JNIEnv *env, jobject obj) {
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of RootKey (RegistryKey.root) object
   jfieldID fid_root = env->GetFieldID(CRegistryKey, "root", registryDescriptor::jfd_RootKey);
   if(fid_root == NULL) { return 0; }

   // get reference to RootKey object
   jobject obj_RootKey = env->GetObjectField(obj, fid_root);
   if(obj_RootKey == NULL) { return 0; }

   // get reference to RootKey class
   jclass CRootKey = env->GetObjectClass(obj_RootKey);
   if(CRootKey == NULL) { return 0; }

   // lookup RootKey.getValue() method
   jmethodID mid_getValue = env->GetMethodID(CRootKey, "getValue", "()I");
   if(mid_getValue == NULL) { return 0; }

   // call RootKey.getValue() to get value
   return env->CallIntMethod(obj_RootKey, mid_getValue);
} // getRootKey()

/**
 * RegistryKey::exists()
 *    Tests if this registry key, as defined by the RootKey and path, currently
 *    exists in the system registry.
 * @return
 */
JNIEXPORT jboolean JNICALL Java_ca_beq_util_win32_registry_RegistryKey_exists(JNIEnv *env, jobject obj) {
   bool bKeyExists = false; // return value

   // step 1: get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // step 2: get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return false; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return false; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return false; }

   // step 3: attempt to open specified key; success means key exists, failure means it doesn't
   //         silently hide failure, simply return false

   // open specified registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) == ERROR_SUCCESS) {
      // key exists, close handle and set return value
      RegCloseKey(hkey);
      bKeyExists = true;
   } // if

   // release native string
   env->ReleaseStringUTFChars(path, szPath);

   return bKeyExists;
} // RegistryKey::exists()

/**
 * RegistryKey::create()
 *    Creates this registry key in the system registry.
 */
JNIEXPORT void JNICALL Java_ca_beq_util_win32_registry_RegistryKey_create(JNIEnv *env, jobject obj) {
   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return; }

   // attempt to creat key in the registry
   HKEY hkey;
   DWORD dwDisposition;
   if(RegCreateKeyEx((HKEY)nRootKey, szPath, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hkey, &dwDisposition) != ERROR_SUCCESS) {
      //RegCloseKey(hkey);
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error creating registry key");
   } // if
   else {
      RegCloseKey(hkey);
      if(dwDisposition == REG_OPENED_EXISTING_KEY) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Attempted to create existing key");
      } // if
   } // else

   env->ReleaseStringUTFChars(path, szPath);
   return;
} // RegistryKey::create()

/**
 * RegistryKey::delete()
 *    Deletes this registry key from the system registry.  The key must not
 *    possess either values or subkeys.
 */
JNIEXPORT void JNICALL Java_ca_beq_util_win32_registry_RegistryKey_delete(JNIEnv *env, jobject obj) {
   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return; }

   if(SHDeleteKey((HKEY)nRootKey, szPath) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error deleting registry key");
   } // if

   env->ReleaseStringUTFChars(path, szPath);
   return;
} // RegistryKey::delete()

/**
 * RegistryKey::hasSubkeys()
 *    Tests if this registry key possesses subkeys.  Use subkeys() to retrieve
 *    an iterator for available subkeys.
 */
JNIEXPORT jboolean JNICALL Java_ca_beq_util_win32_registry_RegistryKey_hasSubkeys(JNIEnv *env, jobject obj) {
   bool bHasSubkeys = false;

   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return false; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return false; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return false; }

   // open specified registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
   } // if
   else {
      // count sub keys
      DWORD dwCount;
      DWORD dwMaxLen;
      if(RegQueryInfoKey(hkey, NULL, NULL, NULL, (LPDWORD)&dwCount, (LPDWORD)&dwMaxLen, NULL, NULL, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error counting registry subkeys");
         return false;
      } // if

      if(dwCount > 0) {
         bHasSubkeys = true;
      } // if

      // key exists, close handle and set return value
      RegCloseKey(hkey);
   } // else

   env->ReleaseStringUTFChars(path, szPath);
   return bHasSubkeys;
} // RegistryKey::hasSubkeys()

/**
 * RegistryKey::hasValues()
 *    Tests if this registry key possess any values.
 */
JNIEXPORT jboolean JNICALL Java_ca_beq_util_win32_registry_RegistryKey_hasValues(JNIEnv *env, jobject obj) {
   bool bHasValues = false;

   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return false; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return false; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return false; }

   // open specified registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
   } // if
   else {
      // count sub keys
      DWORD dwCount;
      DWORD dwMaxLen;
      if(RegQueryInfoKey(hkey, NULL, NULL, NULL, NULL, NULL, (LPDWORD)&dwCount, (LPDWORD)&dwMaxLen, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error counting registry values");
         return false;
      } // if

      // key exists, close handle and set return value
      RegCloseKey(hkey);
      bHasValues = true;
   } // else

   env->ReleaseStringUTFChars(path, szPath);
   return bHasValues;
} // RegistryKey::hasValues()

/**
 * RegistryKey::hasValue()
 *    Tests if this registry key possess the specified value.
 */
JNIEXPORT jboolean JNICALL Java_ca_beq_util_win32_registry_RegistryKey_hasValue(JNIEnv *env, jobject obj, jstring name) {
   bool bHasValue = false;

   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return false; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return false; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return false; }

   // open specified registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
   } // if
   else {
      const char *szName = env->GetStringUTFChars(name, NULL);
      if(RegQueryValueEx(hkey, szName, NULL, NULL, NULL, NULL) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error retreiving registry key information");
      } // if
      else {
         bHasValue = true;
      } // else

      env->ReleaseStringUTFChars(name, szName);
      RegCloseKey(hkey);
   } // else

   env->ReleaseStringUTFChars(path, szPath);
   return bHasValue;
} // RegistryKey::hasValue()

/**
 * RegistryKey::getValue()
 *    Returns a RegistryValue representing the specified value.
 */
JNIEXPORT jobject JNICALL Java_ca_beq_util_win32_registry_RegistryKey_getValue(JNIEnv *env, jobject obj, jstring name) {
   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return NULL; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return NULL; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return NULL; }

   // open registry key for reading
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
      env->ReleaseStringUTFChars(path, szPath);
      return NULL;
   } // if
   else {

      const char *szName = env->GetStringUTFChars(name, NULL);

      // find the type and size of the value
      DWORD type;
      DWORD size;
      if(RegQueryValueEx(hkey, szName, NULL, &type, (LPBYTE)NULL, &size) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error retreiving registry value");
         RegCloseKey(hkey);
         env->ReleaseStringUTFChars(name, szName);
         return NULL;
      } // if

      // get memory to hold the value
      char *data = (char*)malloc(size);

      // read the value
      if(RegQueryValueEx(hkey, szName, NULL, &type, (LPBYTE)data, &size) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error retreiving registry value");
         free(data);
         RegCloseKey(hkey);
         env->ReleaseStringUTFChars(name, szName);
         return NULL;
      } // if

      RegCloseKey(hkey);
      env->ReleaseStringUTFChars(name, szName);

      char* szField;                      // set to string version of registry type
      jobject dataValue;                  // data to be passed in RegistryValue.setData()
      jclass CInteger;                    // java.lang.Integer for DWORD base types
      jmethodID mid_IntegerConstructor;   // java.lang.Integer.Integer(int) for DWORD base types
      char *tmp;                          // buffer, used when expanding REG_EXPAND_SZ types
      DWORD required;                     // buffer size, used when expanding REG_EXPAND_SZ types

      // convert the registry data to a Java type
      switch(type) {
         // raw, unformatted, binary data
         case REG_NONE:
         case REG_BINARY:
            if(type == REG_NONE) {
               szField = "REG_NONE";
            } // if
            else {
               szField = "REG_BINARY";
            } // else

            dataValue = env->NewByteArray(size);
            if(dataValue == NULL) { return NULL; }

            env->SetByteArrayRegion((jbyteArray)dataValue, 0, size, (signed char*)data);
            break;

         // integer/long values
         case REG_DWORD:
         case REG_DWORD_BIG_ENDIAN:
            if(type == REG_DWORD) {
               szField = "REG_DWORD";
            } // if
            else {
               szField = "REG_DWORD_BIG_ENDIAN";
            } // else

            CInteger = env->FindClass(classDescriptor::jcd_Integer);
            if(CInteger == NULL) { return NULL; }

            mid_IntegerConstructor = env->GetMethodID(CInteger, "<init>", "(I)V");
            if(mid_IntegerConstructor == NULL) { return NULL; }

            dataValue = env->NewObject(CInteger, mid_IntegerConstructor, *(long*)data);
            break;

         // string values
         case REG_SZ:
            szField = "REG_SZ";
            dataValue = env->NewStringUTF(data);
            break;

         // expanded environment variable values
         case REG_EXPAND_SZ:
            szField = "REG_EXPAND_SZ";

            // determine required size
            required = ExpandEnvironmentStrings(data, NULL, 0);
            tmp = (char*)malloc(required);

            // expand the strings
            if(!ExpandEnvironmentStrings(data, tmp, required)) {
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error expanding registry strings");
            } // if

            dataValue = env->NewStringUTF(tmp);
            break;

         // string arrays (unsupported)
         case REG_MULTI_SZ:
            szField = "REG_MULTI_SZ";
            env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Unsupported data type");
            break;

         // everything else (nothing)
         default:
            env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Unsupported data type");
            break;
      } // switch

      // look up ValueType class
      jclass CValueType = env->FindClass(registryDescriptor::jcd_ValueType);
      if(CValueType == NULL) { return NULL; }

      // get the appropriate ValueType
      jfieldID fid_type = env->GetStaticFieldID(CValueType, szField, registryDescriptor::jfd_ValueType);
      if(fid_type == NULL) { return NULL; }
      jobject typeValue = env->GetStaticObjectField(CValueType, fid_type);

      // create a new RegistryValue object:
      // get RegistryValue class definition
      jclass CRegistryValue = env->FindClass(registryDescriptor::jcd_RegistryValue);
      if(CRegistryValue == NULL) { return NULL; }

      // use default constructor to create new RegistryValue
      jmethodID mid_Constructor = env->GetMethodID(CRegistryValue, "<init>", "()V");
      if(mid_Constructor == NULL) { return NULL; }
      jobject registryValue = env->NewObject(CRegistryValue, mid_Constructor);

      // use RegistryValue.setName() to set the name
      jmethodID mid_setName = env->GetMethodID(CRegistryValue, "setName", "(Ljava/lang/String;)V");
      if(mid_setName == NULL) { return NULL; }
      env->CallObjectMethod(registryValue, mid_setName, name);

      // use RegistryValue.setType() to set the type
      jmethodID mid_setType = env->GetMethodID(CRegistryValue, "setType", "(Lca/beq/util/win32/registry/ValueType;)V");
      if(mid_setType == NULL) { return NULL; }
      env->CallObjectMethod(registryValue, mid_setType, typeValue);

      // use RegistryValue.setData() to set the data
      jmethodID mid_setData = env->GetMethodID(CRegistryValue, "setData", "(Ljava/lang/Object;)V");
      if(mid_setData == NULL) { return NULL; }
      env->CallObjectMethod(registryValue, mid_setData, dataValue);

      //free(data);
      return registryValue;
   } // else
} // RegistryKey::getValue()

/**
 * RegistryKey::setValue()
 *    Sets the properties of a registry value according to the properties of
 *    the specified RegistryValue.  If the specified value exists, it will be
 *    modified; if not, it will be created.
 */
JNIEXPORT void JNICALL Java_ca_beq_util_win32_registry_RegistryKey_setValue(JNIEnv *env, jobject obj, jobject value) {
   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return; }

   // open registry key for writing
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_ALL_ACCESS, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
      env->ReleaseStringUTFChars(path, szPath);
      return;
   } // if
   else {
      env->ReleaseStringUTFChars(path, szPath);

      // get reference to RegistryValue class
      jclass CRegistryValue = env->GetObjectClass(value);
      if(CRegistryValue == NULL) { return; }

      // execute RootKey.getName() method to get value name
      jmethodID mid_getName = env->GetMethodID(CRegistryValue, "getName", "()Ljava/lang/String;");
      if(mid_getName == NULL) { return; }
      jstring name = (jstring)env->CallObjectMethod(value, mid_getName);

      // execute RootKey.getType() method to get registry data type
      jmethodID mid_getType = env->GetMethodID(CRegistryValue, "getType", "()Lca/beq/util/win32/registry/ValueType;");
      if(mid_getType == NULL) { return; }
      jobject type = env->CallObjectMethod(value, mid_getType);

      // execute RootKey.getData() method to get the value data
      jmethodID mid_getData = env->GetMethodID(CRegistryValue, "getData", "()Ljava/lang/Object;");
      if(mid_getData == NULL) { return; }
      jobject data = env->CallObjectMethod(value, mid_getData);

      // get reference to ValueType class
      jclass CValueType = env->GetObjectClass(type);
      if(CValueType == NULL) { return; }

      // execute ValueType.getValue() method to get the integer value for this data type
      jmethodID mid_getValue = env->GetMethodID(CValueType, "getValue", "()I");
      if(mid_getValue == NULL) { return; }
      int nValue = env->CallIntMethod(type, mid_getValue);

      const char *szName = env->GetStringUTFChars(name, NULL);

      switch(nValue) {
         case REG_NONE:
         case REG_BINARY:
            if(env->IsAssignableFrom(env->GetObjectClass(data), env->FindClass(fieldDescriptor::jfd_byteArray))) {
               const char* dataValue = (char*)env->GetByteArrayElements((jbyteArray)data, NULL);
               DWORD size = env->GetArrayLength((jarray)data);

               if(RegSetValueEx(hkey, szName, 0, (DWORD)nValue, (LPBYTE)dataValue, size) != ERROR_SUCCESS) {
                  env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error setting registry value");
               } // if

               env->ReleaseByteArrayElements((jbyteArray)data, (signed char*)dataValue, 0);
            } // if
            else {
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Incompatible data types");
            } // else
            break;

         case REG_DWORD:
         case REG_DWORD_BIG_ENDIAN:
            if(env->IsAssignableFrom(env->GetObjectClass(data), env->FindClass(classDescriptor::jcd_Integer))) {
               jmethodID mid_intValue = env->GetMethodID(env->GetObjectClass(data), "intValue", "()I");
               int ivalue = env->CallIntMethod(data, mid_intValue);
               const char* dataValue = (char*)&ivalue;
               int size = 4;

               if(RegSetValueEx(hkey, szName, 0, (DWORD)nValue, (LPBYTE)dataValue, size) != ERROR_SUCCESS) {
                  env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error setting registry value");
               } // if
            } // if
            else {
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Incompatible data types");
            } // else
            break;

         case REG_SZ:
         case REG_EXPAND_SZ:
            if(env->IsAssignableFrom(env->GetObjectClass(data), env->FindClass(classDescriptor::jcd_String))) {
               const char* dataValue = env->GetStringUTFChars((jstring)data, NULL);
               int size = env->GetStringLength((jstring)data) + 1;

               if(RegSetValueEx(hkey, szName, 0, (DWORD)nValue, (LPBYTE)dataValue, size) != ERROR_SUCCESS) {
                  env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error setting registry value");
               } // if

               env->ReleaseStringUTFChars((jstring)data, dataValue);
            } // if
            else {
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Incompatible data types");
            } // else
            break;

         case REG_MULTI_SZ:
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Not implemented");
            break;

         default:
               env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Unsupported value type");
            break;
      } // switch

      env->ReleaseStringUTFChars((jstring)name, szName);
      RegCloseKey(hkey);
   } // else
} // RegistryKey::setValue()

/**
 * RegistryKey::deleteValue()
 *    Deletes the specified value from this registry key.
 */
JNIEXPORT void JNICALL Java_ca_beq_util_win32_registry_RegistryKey_deleteValue(JNIEnv *env, jobject obj, jstring name) {
   // get the root key (RootKey) from the RegistryKey (this.getRootKey.getValue())
   int nRootKey = getRootKey(env, obj);

   // get the path (String) from the RegistryKey (this.getPath())
   // get reference to RegistryKey class
   jclass CRegistryKey = env->GetObjectClass(obj);

   // get field ID of path string
   jfieldID fid_path = env->GetFieldID(CRegistryKey, "path", fieldDescriptor::jfd_String);
   if(fid_path == NULL) { return; }

   // get reference to path string
   jstring path = (jstring)env->GetObjectField(obj, fid_path);
   if(path == NULL) { return; }

   // convert to native string
   const char *szPath = env->GetStringUTFChars(path, NULL);
   if(szPath == NULL) { return; }

   // open specified registry key
   HKEY hkey;
   if(RegOpenKeyEx((HKEY)nRootKey, szPath, 0, KEY_READ, &hkey) != ERROR_SUCCESS) {
      env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error opening registry key");
   } // if
   else {
      const char *szName = env->GetStringUTFChars(name, NULL);
      if(RegDeleteValue(hkey, szName) != ERROR_SUCCESS) {
         env->ThrowNew(env->FindClass(registryDescriptor::jcd_RegistryException), "Error deleting registry key value");
      } // if
      env->ReleaseStringUTFChars(name, szName);
      RegCloseKey(hkey);
   } // else

   env->ReleaseStringUTFChars(path, szPath);
} // RegistryKey::deleteValue()
