/* RegistryKeyTest.java
 *
 * jRegistryKey - A JNI wrapper of the Windows Registry functions.
 * Copyright (c) 2001, BEQ Technologies Inc.
 * #205, 3132 Parsons Road
 * Edmonton, Alberta
 * T6N 1L6 Canada
 * (780) 430-0056
 * (780) 437-6121 (fax)
 * http://www.beq.ca
 *
 * Original Author: Joe Robinson <joe@beq.ca>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package ca.beq.util.win32.registry;

import java.util.Arrays;
import java.util.Iterator;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * <code>RegistryKeyTest</code> tests the <code>RegistryKey</code> class.
 *
 * @see ca.beq.util.win32.registry.RegistryKey
 *
 * @author BEQ Technologies Inc.
 * @version 1.0
 */
public class RegistryKeyTest extends TestCase {
   private static final String TESTKEY = "jRegistryKeyTestKey";

   /**
    * Executable entry point.
    */
   public static void main(String[] args) {
      junit.textui.TestRunner.run(suite());
   } // main()

   /**
    * Constructor.
    */
   public RegistryKeyTest (String name) {
      super(name);
   } // RegistryKeyTest()

   /**
    * Returns list of available tests.
    */
   public static Test suite() {
      return new TestSuite(RegistryKeyTest.class);
   } // suite()

   /**
    * Tests that the <code>exists()</code> method succeeds on existing key
    */
   public void testExistsValid() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_LOCAL_MACHINE, "HARDWARE");
      assertEquals("Valid key \"HKLM\\HARDWARE\" must report true (exists)!", true, r.exists());
   }

   /**
    * Tests that the <code>exists()</code> method fails on a non-existant key
    */
   public void testExistsInvalid() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_LOCAL_MACHINE, "GoodBadImTheOneWithTheGun");
      assertEquals("Bogus key \"HKCU\\OogaBooga\" must report false (does not exist)!", false, r.exists());
   } // testExistsInvalid()

   /**
    * Tests creating a non-existant key succeeds
    */
   public void testCreateNew() {
      try {
         RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
         if(r.exists()) {
            r.delete();
         } // if

         r.create();
      } // try
      catch(RegistryException re) {
         fail("Creation of a new key must succeed!");
      } // catch
   } // testCreateNew()

   /**
    * Tests creating a duplicate key fails
    */
   public void testCreateDuplicate() {
      try {
         RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
         if(!r.exists()) {
            r.create();
         } // if

         r.create();
         fail("Creation of a duplicate key must fail!");
      } // try
      catch(RegistryException re) {
      } // catch
   } // testCreateDuplicate()

   /**
    * Tests that deleting a valid key succeeds
    */
   public void testDeleteValid() {
      try {
         RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
         if(!r.exists()) {
            r.create();
         } // if

         r.delete();
      } // try
      catch(RegistryException re) {
         fail("Deletion of a non-empty key must succeed!");
      } // catch
   } // testDeleteValid()

   /**
    * Tests that deleting an invalid key fails
    */
   public void testDeleteInvalid() {
      try {
         RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
         if(r.exists()) {
            r.delete();
         } // if

         r.delete();
         fail("Deletion of an invalid key must fail!");
      } // try
      catch(RegistryException re) {
      } // catch
   } // testDeleteInvalid()

   /**
    * Tests that the <code>hasSubkeys()</code> method fails on a key without
    * subkeys
    */
   public void testHasSubkeysFail() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);

      try {
         if(r.exists()) {
            r.delete();
         } // if

         r.create();
         assertEquals("Checking if a key without subkey has keys must fail!", false, r.hasSubkeys());
      } // try
      catch(RegistryException re) {
         fail("Subkey creation must succeed!");
      } // catch
   } // testHasSubkeysFail()

   /**
    * Tests that the <code>hasSubkeys()</code> method succeeds on a key that
    * has subkeys
    */
   public void testHasSubkeys() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);

      try {
         if(r.exists()) {
            r.delete();
         } // if

         r.create();
         RegistryKey rr = r.createSubkey(TESTKEY);
         assertEquals("Checking if a key with subkey has keys must not fail!", true, r.hasSubkeys());
      } // try
      catch(RegistryException re) {
         fail("Subkey creation must succeed!");
      } // catch
   } // testHasSubkeys()

   /**
    * Test that the <code>setValue()</code> and <code>getValue()</code> methods
    * succeed.
    */
   public void testSetGetValue() {
      String REGVALUE_SZ = "This is a test";
      String REGVALUE_EXPAND_SZ = "PATH=%path%";
      Integer REGVALUE_DWORD = new Integer(1);
      Integer REGVALUE_DWORD_LITTLE_ENDIAN = new Integer(2);
      byte[] REGVALUE_BINARY = new byte[] {0,1,2,3,4,5,6,7,8,9};

      // test REG_SZ
      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
      RegistryValue v = new RegistryValue("test1", ValueType.REG_SZ, REGVALUE_SZ);
      r.setValue(v);
      assertEquals("REG_SZ strings must match!", REGVALUE_SZ, (String)r.getValue("test1").getData());

      // test REG_EXPAND_SZ
      v = new RegistryValue("test2", ValueType.REG_EXPAND_SZ, REGVALUE_EXPAND_SZ);
      r.setValue(v);

      // test REG_DWORD
      v = new RegistryValue("test3", ValueType.REG_DWORD, REGVALUE_DWORD);
      r.setValue(v);
      assertEquals("REG_DWORD values must match!", REGVALUE_DWORD.intValue(), ((Integer)r.getValue("test3").getData()).intValue());

      // test REG_DWORD_LITTLE_ENDIAN
      v = new RegistryValue("test4", ValueType.REG_DWORD_LITTLE_ENDIAN, REGVALUE_DWORD_LITTLE_ENDIAN);
      r.setValue(v);
      assertEquals("REG_DWORD_LITTLE_ENDIAN values must match!", REGVALUE_DWORD_LITTLE_ENDIAN.intValue(), ((Integer)r.getValue("test4").getData()).intValue());

      v = new RegistryValue("test5", ValueType.REG_BINARY, REGVALUE_BINARY);
      r.setValue(v);
      assertTrue("REG_BINARY values must match!", Arrays.equals((byte[])r.getValue("test5").getData(), REGVALUE_BINARY));
   } // testSetValue()

   /**
    * Test that the <code>subkeys</code> method correctly returns an Iterator.
    */
   public void testSubkeys() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
      assertTrue("subkeys() must return an iterator", r.subkeys() instanceof Iterator);
      assertNotNull("subkeys() must not be null!", r.subkeys());
   } // testSubkeys

   /**
    * Tests that the <code>subkeys</code> method correctly iterates through subkeys.
    */
   public void testKeyIterator() {
      int NUM_KEYS = 10;

      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
      if(r.exists()) {
         r.delete();
      } // if

      r.create();
      for(int index = 0; index < NUM_KEYS; index++) {
         r.createSubkey("subkey-" + index);
      } // for

      if(r.hasSubkeys()) {
         System.out.println("In key " + r.toString());
         int count = 0;
         Iterator i = r.subkeys();
         while(i.hasNext()) {
            RegistryKey rr = (RegistryKey)i.next();
            System.out.println("Found subkey: " + rr.toString());
            count++;
         } // while

         assertEquals("We created these keys - they should exist!", count, NUM_KEYS);
      } // if
      else {
         fail("Subkeys should have been created - this is serious!");
      } // else
   } // testKeyIterator()

   /**
    * Tests that the <code>values</code> method returns an Iterator.
    */
   public void testValues() {
      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
      assertTrue("values() must return an iterator!", r.values() instanceof Iterator);
      assertNotNull("values() must not be null!", r.values());
   } // testValues()

   /**
    * Tests that the <code>values</code> method correctly iterates through values.
    */
   public void testValueIterator() {
      String REGVALUE_SZ = "This is a test";
      String REGVALUE_EXPAND_SZ = "PATH=%path%";
      Integer REGVALUE_DWORD = new Integer(1);
      Integer REGVALUE_DWORD_LITTLE_ENDIAN = new Integer(2);
      byte[] REGVALUE_BINARY = new byte[] {0,1,2,3,4,5,6,7,8,9};

      RegistryKey r = new RegistryKey(RootKey.HKEY_CURRENT_USER, TESTKEY);
      if(r.exists()) {
         r.delete();
      } // if
      r.create();

      RegistryValue v = new RegistryValue("test1", ValueType.REG_SZ, REGVALUE_SZ);
      r.setValue(v);

      v = new RegistryValue("test2", ValueType.REG_EXPAND_SZ, REGVALUE_EXPAND_SZ);
      r.setValue(v);

      v = new RegistryValue("test3", ValueType.REG_DWORD, REGVALUE_DWORD);
      r.setValue(v);

      v = new RegistryValue("test4", ValueType.REG_DWORD_LITTLE_ENDIAN, REGVALUE_DWORD_LITTLE_ENDIAN);
      r.setValue(v);

      v = new RegistryValue("test5", ValueType.REG_BINARY, REGVALUE_BINARY);
      r.setValue(v);

      if(r.hasValues()) {
         System.out.println("In key " + r.toString());
         int count = 0;
         Iterator i = r.values();
         while(i.hasNext()) {
            RegistryValue rv = (RegistryValue)i.next();
            System.out.println("Found value: " + rv.toString());
            count++;
         } // while

         assertEquals("We created these keys - they should exist!", count, 5);
      } // if
      else {
         fail("Values should have been created - this is serious!");
      } // else
   } // testValueIterator()
} // RegistryKey